/*:
 * @target MZ
 * @plugindesc Voice Linker v1.0.0 — SimpleVoice の再生をフックし、「直後の文章1回だけ」バックログに結び付ける。再生も担当。
 * @author Human
 * @help
 * ■役割
 * - AudioManager.playVoice(voice, loop, channel) をフックし、予約（reservation）を保存。
 * - 次の「文章の表示」開始時（BacklogCore が呼ぶ）に reservation を消費して記録。
 * - バックログからの再生（OK押下）は HSBL.playFromEntry で実行。予約には影響させない。
 *
 * ■並び順（推奨）
 * SimpleVoice.js → このファイル → HS_BacklogCore_Simple.js
 */
(() => {
  'use strict';

  // 共有名前空間
  window.HSBL = window.HSBL || {};

  // name 正規化（audio/se/ の除去、拡張子除去）
  function normalize(raw){
    let s = String(raw || '');
    s = s.replace(/^audio[\\\/]+se[\\\/]+/i, '');
    s = s.replace(/\.(ogg|m4a|wav)$/i, '');
    return s;
  }

  const HSBL = window.HSBL;
  HSBL._reservation = null;   // {name, volume, pitch, pan, loop, channel}
  HSBL._guard = false;        // backlog再生中ガード

  HSBL.consumeReservation = function(){
    const r = HSBL._reservation;
    HSBL._reservation = null;
    return r ? { ...r } : null;
  };

  // バックログエントリからの再生
  HSBL.playFromEntry = function(entry){
    if (!entry || !entry.voice) return;
    const v = entry.voice;
    HSBL._guard = true; // これにより、下のフックで新たな予約が立たない
    try{
      if (AudioManager.playVoice){
        AudioManager.playVoice(
          { name: v.name, volume: v.volume, pitch: v.pitch, pan: v.pan },
          !!v.loop,
          +v.channel || 0
        );
      } else {
        AudioManager.playSe({ name: v.name, volume: v.volume ?? 90, pitch: v.pitch ?? 100, pan: v.pan ?? 0 });
      }
    } catch(e){
      console.warn('[HSBL-Linker] playFromEntry failed:', e);
    } finally {
      HSBL._guard = false;
    }
  };

  // SimpleVoice あり：playVoice をフック
  if (AudioManager.playVoice){
    const _playVoice = AudioManager.playVoice;
    AudioManager.playVoice = function(voice, loop, channel){
      const res = _playVoice.apply(this, arguments);
      try{
        if (!HSBL._guard && voice && voice.name != null){
          HSBL._reservation = {
            name:   normalize(voice.name),
            volume: +(voice.volume ?? 100),
            pitch:  +(voice.pitch  ?? 100),
            pan:    +(voice.pan    ??   0),
            loop:   !!loop,
            channel:+(channel ?? 0)
          };
        }
      }catch(_){}
      return res;
    };
  } else {
    // SimpleVoiceが無い環境への保険：SE再生を拾う
    const _playSe = AudioManager.playSe;
    AudioManager.playSe = function(se){
      const res = _playSe.apply(this, arguments);
      try{
        if (!HSBL._guard && se && se.name != null){
          HSBL._reservation = {
            name:   normalize(se.name),
            volume: +(se.volume ?? 90),
            pitch:  +(se.pitch  ?? 100),
            pan:    +(se.pan    ??   0),
            loop:   false,
            channel:0
          };
        }
      }catch(_){}
      return res;
    };
  }

})();
